﻿using System;
using System.Collections.Generic;
using System.Data;
using System.Data.SqlClient;
using System.Globalization;

namespace gov.va.med.vbecs.Common.Configuration
{
    /// <summary>
    /// AppSettingsReader that reads configuration values from DB
    /// </summary>
    class AppSettingsReaderMsSql : IAppSettingsReader
    {
        /// <summary>
        /// Application name. Assigned by Spring.NET
        /// </summary>
        public string AppName { get; internal set; }

        /// <summary>
        /// Indicates if it is a service. Assigned by Spring.NET
        /// </summary>
        public bool IsService { get; internal set; }

        /// <summary>
        /// DataSourceString
        /// </summary>
        public string DataSourceString { get; internal set; }

        /// <summary>
        /// SettingsMap
        /// </summary>
        protected IDictionary<string, string> SettingsMap;

        /// <summary>
        /// AppSettingsReaderMsSql constructor
        /// </summary>
        internal AppSettingsReaderMsSql()
        {}

        /// <summary>
        /// GetValue
        /// </summary>
        /// <param name="key"></param>
        /// <param name="type"></param>
        /// <returns></returns>
        public object GetValue(string key, Type type)
        {
            if (key == null) throw new ArgumentNullException("key");

            if (SettingsMap == null)
                SettingsMap = read_application_settings(AppName);

           
            if (!SettingsMap.ContainsKey(key))
                throw new ArgumentException(string.Format("Provided Key is not found in the configuration: {0}, for application: {1}", key, AppName));

            var val = SettingsMap[key];
            if (val == null)
            {
                // Only true if T is a value type 
                if (type.IsValueType)
                    // Since value types can't be NULL, throw exception
                    throw new InvalidOperationException(string.Format(
                        "Value for [{0}] can't be null", key));
                return null;
            }
            try{
                return Convert.ChangeType(val, type, CultureInfo.InvariantCulture);
            }
            catch (Exception)
            {
                throw new InvalidOperationException(string.Format(
                    "Can't parse [{0}] string for Key=[{1}] to Type=[{2}]", val, key, type));
            }
        }


        private IDictionary<string, string> read_application_settings(string appName)
        {
            SettingsMap = new Dictionary<string, string>();

            SqlParameter[] prms =
                {
                    new SqlParameter(VbecsStoredProcs.GetAllVbecsGlobalConfigValue.ApplicationName, SqlDbType.VarChar)
                };
            prms[0].Value = appName;

            var dtParam =
                StoredProcedure.GetData(VbecsStoredProcs.GetAllVbecsGlobalConfigValue.StoredProcName, prms).Tables[0];

            foreach (DataRow row in dtParam.Rows)
            {
                var configKey = row[VbecsTables.VbecsGlobalConfig.ParameterName] as string;
                var configValue = row[VbecsTables.VbecsGlobalConfig.ParameterValue];
                SettingsMap.Add(new KeyValuePair<string, string>(configKey,
                                                                 (configValue == null || configValue == DBNull.Value) ? null : configValue.ToString()));
            }

            return SettingsMap;
        }
    }
}
